// Polygon Paste.js
//
// 2014-05-07: firset release
//
// Hiroto Tsubaki tg@tres-graficos.jp
//

function buildUI( tool ) {
	tool.addParameterSeparator("Polygon Paste");
	
	tool.addParameterLink("target polygon", false);
	
	tool.addParameterSelector("direction coord.", ["object", "world"], false, false);
	tool.addParameterSelector("paste direction", ["x", "y", "z"], false, false);
	
	tool.addParameterFloat("paste offset", 0, -1000, 1000, false, false);
	
	tool.addParameterButton("paste", "apply", "pastePolygon");
	
}

function pastePolygon( tool ) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	var poly = tool.getParameter("target polygon");
	
	if (!obj || !poly) {
		OS.beep(); return;
	}
	
	if (obj.type() == POLYGONOBJ && poly.family() == NGONFAMILY) {
		var pi, ei, i, j, k;
		
		var core = obj.core();
		var objMat = obj.obj2WorldMatrix();
		var objMatInverse = objMat.inverse();
		var objRot = obj.getParameter("rotation");
		var objRotMat = new Mat4D( ROTATE_HPB, objRot.x, objRot.y, objRot.z );
		
		var polyCore = poly.modCore();
		var polyMat = poly.obj2WorldMatrix();
		
		print('-- Polygon Paste.js');
		
		var dir = tool.getParameter("paste direction");
		var coord = tool.getParameter("direction coord.");
		var offset = tool.getParameter("paste offset");
		
		var dirVec;
		switch( parseInt(dir) ) {
			case 0: // -x
				dirVec = new Vec3D( -1, 0, 0 );
				break;
			case 1: // -y
				dirVec = new Vec3D( 0, -1, 0 );
				break;
			case 2: // -z
				dirVec = new Vec3D( 0, 0, -1 );
				break;
		}
		// obj vector
		if (coord == 0) {
			dirVec = objRotMat.multiply( dirVec );
		}
		dirVec = Vec3D_normalize( dirVec );
		
		//print( 'dirVec:' + Vec3D_toString( dirVec ) );
		
		// undoable
		obj.recordGeometryForUndo();
		
		var guidePolygonCount = polyCore.polygonCount();
		var hitInfo = {};
		
		var vertexCount = core.vertexCount();
		for (i = 0;i < vertexCount;i++) {
		  var vertex = objMat.multiply( core.vertex( i ) );
		  
		  hitInfo[ i ] = [];
		  
		  for (j = 0;j < guidePolygonCount;j++) {
		    var guidePolygonSize = polyCore.polygonSize(j);
		    var normal = polyCore.normal( j );
		    
		    for (k = 0;k < guidePolygonSize - 2;k++) {
		      var triangles = polyCore.triangle( j, k );
		      var t0 = polyCore.vertex( polyCore.vertexIndex( j, triangles[0] ) );
		      var t1 = polyCore.vertex( polyCore.vertexIndex( j, triangles[1] ) );
		      var t2 = polyCore.vertex( polyCore.vertexIndex( j, triangles[2] ) );
		      
					t0 = polyMat.multiply( t0 ); t1 = polyMat.multiply( t1 ); t2 = polyMat.multiply( t2 );
					
					var crossPoint = triangleIntersect( vertex, dirVec, t0, t1, t2 );
					
					if (crossPoint) {
					  hitInfo[ i ].push( objMatInverse.multiply( crossPoint.add( normal.multiply( offset ) ) ) );
					}
		    }
		    
		  }
		  
	    if (hitInfo[i].length < 1) {
				  //print( 'vertex:' + i + ' unhit' );
	    }
		}
		
		for (i in hitInfo) {
		  var info = hitInfo[i];
		  
		  if (info.length > 0) {
  		  var baseVertex = core.vertex( parseInt( i ) );
  		  
  		  var vertex = info[0];
  		  var distance = Vec3D_distance( baseVertex, vertex );
  		  
  		  for (j = 1;j < info.length;j++) {
  		    var vertex_comp = info[j];
  		    var distance_comp = Vec3D_distance( baseVertex, vertex_comp);
  		    if (distance > distance_comp) {
  		      vertex = info[j];
  		      distance = distance_comp;
  		    }
  		  }
  		  
  		  core.setVertex( parseInt( i ), vertex );
		  }
		}
		obj.update();
	}
}

// Tomas Möller 
var triangleIntersect = function( orig, dir, v0, v1, v2 ) {
	var e1, e2, pvec, tvec, qvec;
	var epsilon = 1e-12;
	var det;
	var t, u, v;
	var inv_det;
	
	e1 = v1.sub( v0 );
	e2 = v2.sub( v0 );
	
	pvec = dir.cross( e2 );
	det = e1.dot( pvec );
	det = (det.norm)? det.x : det;
	
	if (det > epsilon) {
		tvec = orig.sub( v0 );
		u = tvec.dot( pvec );
		u = (u.norm)? u.x : u;
		if (u < 0 || u > det) return false;
		
		qvec = tvec.cross( e1 );
		
		v = dir.dot( qvec );
		v = (v.norm)? v.x : v;
		if (v < 0 || u + v > det) return false; 
	} else if (det < -epsilon) {
		tvec = orig.sub( v0 );
		
		u = tvec.dot( pvec );
		u = (u.norm)? u.x : u;
		if (u > 0 || u < det) return false;
		
		qvec = tvec.cross( e1 );
		
		v = dir.dot( qvec );
		v = (v.norm)? v.x : v;
		if (v > 0 || u + v < det) return false;
	} else {
		return false;
	}
	
	inv_det = 1 / det;
	
	t = e2.dot( qvec );
	t = (t.norm)? t.x : t;
	t = t * inv_det;
	
	u = u * inv_det;
	v = v * inv_det;
	
	var crossPoint = orig.add( dir.multiply( t ) );
	
	return crossPoint;
}

var Vec3D_distance = function() {
	if( arguments.length == 1)
			return Math.sqrt( arguments[0].x*arguments[0].x + arguments[0].y*arguments[0].y + arguments[0].z*arguments[0].z );
	var p = arguments[1].sub(arguments[0]);
	return Math.sqrt( p.x*p.x + p.y*p.y + p.z*p.z );
}

var Vec3D_normalize = function(vec) {
	var l = vec.norm();
	if (l != 0) {
		return vec.multiply( 1/l );
	}
	return vec;
}

var Vec3D_toString = function(vec) {
  return (vec.x.toFixed(3) + ', ' + vec.y.toFixed(3) + ', ' + vec.z.toFixed(3));
}

Array.prototype.pushUnique = function( val ) {
	var len = this.length;
	for (var i = 0;i < len;i++) {
		var data = this[i];
		if (data == val) {
			return false;
		}
	}
	this.push( val );
}

Array.prototype.toString = function() {
	var len = this.length;
	var str = '[';
	for (var i =0;i < len;i++) {
		str += this[i]
		if (i != len-1) str += ', ';
	}
	return str + ']';
}
